<?php

namespace App\Http\Controllers\Admin;

use App\Http\Controllers\Controller;
use App\Models\Video;
use Illuminate\Http\Request;
use Illuminate\Support\Facades\Storage;

class VideoController extends Controller
{
    /**
     * Display a listing of the resource.
     */
    public function index()
    {
        $videos = Video::orderBy('sort_order')->get();
        return view('admin.videos.index', compact('videos'));
    }

    /**
     * Show the form for creating a new resource.
     */
    public function create()
    {
        return view('admin.videos.create');
    }

    /**
     * Store a newly created resource in storage.
     */
    public function store(Request $request)
    {
        $rules = [
            'title' => 'nullable|string|max:255',
            'video_type' => 'required|in:link,upload',
            'sort_order' => 'nullable|integer',
        ];

        if ($request->video_type === 'link') {
            $rules['url'] = 'required|url';
        } else {
            $rules['video_file'] = 'required|file|mimes:mp4,mov,avi,wmv|max:102400'; // Max 100MB
        }

        $request->validate($rules);

        $data = [
            'title' => $request->title,
            'video_type' => $request->video_type,
            'sort_order' => $request->sort_order ?? 0,
        ];

        if ($request->video_type === 'link') {
            $data['url'] = $request->url;
        } else {
            $path = $request->file('video_file')->store('videos', 'public');
            $data['video_path'] = $path;
        }

        Video::create($data);

        return redirect()->route('admin.videos.index')->with('message', 'Video added successfully!');
    }

    /**
     * Display the specified resource.
     */
    public function show(string $id)
    {
        // Not used
    }

    /**
     * Show the form for editing the specified resource.
     */
    public function edit(Video $video)
    {
        return view('admin.videos.edit', compact('video'));
    }

    /**
     * Update the specified resource in storage.
     */
    public function update(Request $request, Video $video)
    {
        $rules = [
            'title' => 'nullable|string|max:255',
            'video_type' => 'required|in:link,upload',
            'sort_order' => 'nullable|integer',
        ];

        if ($request->video_type === 'link') {
            $rules['url'] = 'required|url';
        } else {
            $rules['video_file'] = 'nullable|file|mimes:mp4,mov,avi,wmv|max:102400'; // Max 100MB
        }

        $request->validate($rules);

        $data = [
            'title' => $request->title,
            'video_type' => $request->video_type,
            'sort_order' => $request->sort_order ?? 0,
        ];

        if ($request->video_type === 'link') {
            $data['url'] = $request->url;
            $data['video_path'] = null;

            // Delete old video file if exists
            if ($video->video_path && Storage::disk('public')->exists($video->video_path)) {
                Storage::disk('public')->delete($video->video_path);
            }
        } else {
            if ($request->hasFile('video_file')) {
                // Delete old video file if exists
                if ($video->video_path && Storage::disk('public')->exists($video->video_path)) {
                    Storage::disk('public')->delete($video->video_path);
                }

                $path = $request->file('video_file')->store('videos', 'public');
                $data['video_path'] = $path;
            }
            $data['url'] = null;
        }

        $video->update($data);

        return redirect()->route('admin.videos.index')->with('message', 'Video updated successfully!');
    }

    /**
     * Remove the specified resource from storage.
     */
    public function destroy(Video $video)
    {
        // Delete video file if exists
        if ($video->video_path && Storage::disk('public')->exists($video->video_path)) {
            Storage::disk('public')->delete($video->video_path);
        }

        $video->delete();

        return redirect()->route('admin.videos.index')->with('message', 'Video deleted successfully!');
    }

    public function massDestroy(Request $request)
    {
        Video::whereIn('id', request('ids'))->delete();

        return response(null, 204);
    }
}
